<?php

namespace Modules\Store\Http\Controllers;

use App\City;
use App\Country;
use App\State;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Setting\Model\GeneralSetting;
use Modules\Store\Http\Requests\PickupLocationRequest;
use Modules\Store\Repositories\PickupLocationRepository;

class PickupLocationController extends Controller
{
    protected $pickupLocationRepo;

    public function __construct(PickupLocationRepository $pickupLocationRepo)
    {
        $this->pickupLocationRepo = $pickupLocationRepo;
    }

    public function index()
    {
        try {
            $countryId = Settings('country_id');
            $data['pickup_locations'] = $this->pickupLocationRepo->all();
            $data['countries'] = Country::where('active_status', 1)->get();
            $data['states'] = State::where('country_id', $countryId)->orderBy('name')->get();
            $state_id = $data['states'][0]->id;
            $data['cities'] = City::where('state_id', $state_id)->orderBy('name')->get();
            return view('store::shipping.pickup_locations.index', $data);
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json(['error' => $e->getMessage()], 503);
        }
    }

    public function store(PickupLocationRequest $request)
    {
        try {
            $this->pickupLocationRepo->create($request->all());
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('store.pickuplocation.index');
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }
    }

    public function get_states(Request $request)
    {
        return $this->pickupLocationRepo->getStateByCountry($request->get('country_id'));
    }

    public function get_cities(Request $request)
    {
        return $this->pickupLocationRepo->getCityByState($request->get('state_id'));
    }

    public function show($id)
    {
        try {
            $data['row'] = $this->pickupLocationRepo->find($id);
            return view('store::shipping.pickup_locations.components._show', $data);
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json(['error' => $e->getMessage()], 503);
        }
    }


    public function edit($id)
    {
        try {
            $data['edit'] = $this->pickupLocationRepo->find($id);
            $data['setting'] = GeneralSetting::get();
            $countryId = Settings('country_id');
            $data['pickup_locations'] = $this->pickupLocationRepo->all();
            $data['countries'] = Country::where('active_status', 1)->get();
            $data['states'] = State::where('country_id', $countryId)->orderBy('name')->get();
            $data['cities'] = City::where('state_id', $data['edit']['state_id'])->orderBy('name')->get();
            return view('store::shipping.pickup_locations.index', $data);
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json(['error' => $e->getMessage()], 503);
        }
    }


    public function update(PickupLocationRequest $request)
    {
        try {
            $this->pickupLocationRepo->update($request->validated(), $request->id);
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('store.pickuplocation.index');
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }
    }

    public function destroy(Request $request)
    {
        try {
            $result = $this->pickupLocationRepo->delete($request->id);
            if ($result) {
                return $this->reloadWithData();
            } else {
                return response()->json([
                    'msg' => 'Default pickup location is not deletable.'
                ]);
            }
        } catch (\Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }
    }


    public function status(Request $request)
    {
        try {
            $this->pickupLocationRepo->status($request->except("_token"));
            return 1;
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function setDefault(Request $request)
    {
        try {
            $this->pickupLocationRepo->setDefault($request->except("_token"));
            return $this->reloadWithData();
        } catch (\Exception $e) {
            return 0;
        }
    }

    private function reloadWithData()
    {
        try {
            $res = '';
            if (session()->has('ship_rocket_add_location_res')) {
                $res = session()->get('ship_rocket_add_location_res');
            }

            $data['setting'] = GeneralSetting::get();
            $countryId = Settings('country_id');
            $data['pickup_locations'] = $this->pickupLocationRepo->all();
            $data['countries'] = Country::where('active_status', 1)->get();
            $data['states'] = State::where('country_id', $countryId)->orderBy('name')->get();
            $state_id = $data['states'][0]->id;
            $data['cities'] = City::where('state_id', $state_id)->orderBy('name')->get();
            return response()->json([
                'TableData' => (string)view('store::shipping.pickup_locations.components._list', $data),
                'createForm' => (string)view('store::shipping.pickup_locations.components._create', $data),
                'ship_rocket_response' => $res,
            ], 200);
        } catch (\Exception $e) {
            Toastr::error($e->getMessage(), trans('common.Error'));
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }
    }

    public function setPickupLocation($id)
    {
        try {
            $this->pickupLocationRepo->setPickupLocation($id);
            return response()->json(['status' => 200]);

        } catch (\Exception $e) {
            Toastr::error($e->getMessage(), trans('common.Error'));
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }
    }
}
