<?php

namespace Modules\Store\Exports;

use Illuminate\Contracts\View\View;
use Maatwebsite\Excel\Concerns\FromView;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithMultipleSheets;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Events\AfterSheet;
use Modules\Store\Entities\ProductAuthor;
use Modules\Store\Entities\ProductBrand;
use Modules\Store\Entities\ProductCategory;
use Modules\Store\Exports\DropDownSheets\AuthorList;
use Modules\Store\Exports\DropDownSheets\BrandList;
use Modules\Store\Exports\DropDownSheets\CategoryList;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\Cell\DataValidation;

class ProductSampleExport implements WithTitle, WithMultipleSheets, FromView, WithEvents
{
    protected $selects;
    protected $row_count;
    protected $column_count;

    public function sheets(): array
    {
        $sheets = [];
        $sheets[] = new ProductSampleExport();
        $sheets[] = new CategoryList();
        $sheets[] = new BrandList();
        $sheets[] = new AuthorList();
        return $sheets;
    }

    public function __construct()
    {
        $categoryList = new CategoryList();
        $brandList = new BrandList();
        $authorList = new AuthorList();
        $type = ['Virtual', 'Physical'];
        $yesNo = ['Yes', 'No'];
        $selects = [
            ['columns_name' => 'B', 'options' => $type],
            ['columns_name' => 'C', 'options' => [], 'sheet_name' => 'product_categories', 'count' => $categoryList->query()->count()],
            ['columns_name' => 'D', 'options' => [], 'sheet_name' => 'product_brands', 'count' => $brandList->query()->count()],
            ['columns_name' => 'G', 'options' => [], 'sheet_name' => 'product_authors', 'count' => $authorList->query()->count()],
            ['columns_name' => 'J', 'options' => $yesNo],
        ];
        $this->selects = $selects;
        $this->row_count = 1000;
        $this->column_count = 10;
    }

    public function title(): string
    {
        return 'Import';
    }

    public function view(): View
    {
        $category = ProductCategory::first();
        $brand = ProductBrand::first();
        $author = ProductAuthor::first();
        return view('store::products.product_sample_export', compact('category', 'brand', 'author'));
    }


    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $row_count = $this->row_count;
                $column_count = $this->column_count;
                foreach ($this->selects as $select) {
                    $drop_column = $select['columns_name'];
                    $cell = $drop_column . '2';
                    $validation = $event->getSheet()->getDataValidation($cell);

                    $validation->setType(DataValidation::TYPE_LIST);
                    $validation->setAllowBlank(false);
                    $validation->setShowInputMessage(true);
                    $validation->setShowErrorMessage(true);
                    $validation->setShowDropDown(true);
                    $validation->setErrorTitle('Input error');
                    $validation->setError('Value is not in list.');
                    $validation->setPromptTitle('Pick from list');
                    $validation->setPrompt('Please pick a value from the drop-down list.');

                    if (isset($select['sheet_name'])) {
                        $validation->setFormula1("{$select['sheet_name']}!\$A\$1:\$A\${$select['count']}");
                    } else {
                        $options = $select['options'];
                        $validation->setFormula1(sprintf('"%s"', implode(',', $options)));
                    }
                    $validation->setFormula2('');
                    for ($i = 3; $i <= $row_count; $i++) {
                        $event->sheet->getCell("{$drop_column}{$i}")->setDataValidation(clone $validation);
                    }

                    for ($i = 1; $i <= $column_count; $i++) {
                        $column = Coordinate::stringFromColumnIndex($i);
                        $event->sheet->getColumnDimension($column)->setAutoSize(true);
                    }
                }
            },
        ];
    }
}
